<?php

namespace App\Http\Controllers;

use App\Models\ConfiguracionHardware;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

class ConfiguracionHardwareController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = ConfiguracionHardware::query();

        if ($request->has('tipo')) {
            $query->where('tipo', $request->tipo);
        }

        if ($request->has('activo')) {
            $query->where('activo', $request->activo);
        }

        $hardware = $query->orderBy('tipo')->orderBy('nombre')->get();

        return response()->json(['data' => $hardware], JsonResponse::HTTP_OK);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'tipo' => ['required', 'string', 'in:camara_ip,impresora'],
            'nombre' => ['required', 'string', 'max:255'],
            'ip_url' => ['nullable', 'string', 'max:255'],
            'puerto' => ['nullable', 'integer', 'min:1', 'max:65535'],
            'usuario' => ['nullable', 'string', 'max:255'],
            'password' => ['nullable', 'string', 'max:255'],
            'url_snapshot' => ['nullable', 'string', 'max:500'],
            'modelo' => ['nullable', 'string', 'max:255'],
            'configuracion_adicional' => ['nullable', 'json'],
            'activo' => ['boolean'],
        ], [
            'tipo.required' => 'El tipo es obligatorio.',
            'tipo.in' => 'El tipo debe ser "camara_ip" o "impresora".',
            'nombre.required' => 'El nombre es obligatorio.',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }

        $data = $request->all();
        if ($request->has('configuracion_adicional')) {
            $data['configuracion_adicional'] = json_decode($request->configuracion_adicional, true);
        }

        $hardware = ConfiguracionHardware::create($data);

        return response()->json(['data' => $hardware], JsonResponse::HTTP_CREATED);
    }

    public function show(ConfiguracionHardware $configuracionHardware): JsonResponse
    {
        return response()->json(['data' => $configuracionHardware], JsonResponse::HTTP_OK);
    }

    public function update(Request $request, ConfiguracionHardware $configuracionHardware): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'tipo' => ['required', 'string', 'in:camara_ip,impresora'],
            'nombre' => ['required', 'string', 'max:255'],
            'ip_url' => ['nullable', 'string', 'max:255'],
            'puerto' => ['nullable', 'integer', 'min:1', 'max:65535'],
            'usuario' => ['nullable', 'string', 'max:255'],
            'password' => ['nullable', 'string', 'max:255'],
            'url_snapshot' => ['nullable', 'string', 'max:500'],
            'modelo' => ['nullable', 'string', 'max:255'],
            'configuracion_adicional' => ['nullable', 'json'],
            'activo' => ['boolean'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }

        $data = $request->all();
        if ($request->has('configuracion_adicional')) {
            $data['configuracion_adicional'] = json_decode($request->configuracion_adicional, true);
        }

        $configuracionHardware->update($data);

        return response()->json(['data' => $configuracionHardware], JsonResponse::HTTP_OK);
    }

    public function destroy(ConfiguracionHardware $configuracionHardware): JsonResponse
    {
        $configuracionHardware->update(['activo' => false]);
        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }
}

