<?php

namespace App\Http\Controllers;

use App\Models\TipoCarga;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

class TipoCargaController extends Controller
{
    public function index(): JsonResponse
    {
        $tiposCarga = TipoCarga::activos()->get();
        return response()->json(['data' => $tiposCarga], JsonResponse::HTTP_OK);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'codigo' => ['required', 'string', 'max:50', Rule::unique('tipos_carga', 'codigo')],
            'nombre' => ['required', 'string', 'max:255'],
            'descripcion' => ['nullable', 'string'],
            'activo' => ['boolean'],
        ], [
            'codigo.required' => 'El código es obligatorio.',
            'codigo.unique' => 'Ya existe un tipo de carga con este código.',
            'nombre.required' => 'El nombre es obligatorio.',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }

        $tipoCarga = TipoCarga::create($request->all());
        return response()->json(['data' => $tipoCarga], JsonResponse::HTTP_CREATED);
    }

    public function show(TipoCarga $tipoCarga): JsonResponse
    {
        return response()->json(['data' => $tipoCarga], JsonResponse::HTTP_OK);
    }

    public function update(Request $request, TipoCarga $tipoCarga): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'codigo' => [
                'required',
                'string',
                'max:50',
                Rule::unique('tipos_carga', 'codigo')->ignore($tipoCarga->id)
            ],
            'nombre' => ['required', 'string', 'max:255'],
            'descripcion' => ['nullable', 'string'],
            'activo' => ['boolean'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }

        $tipoCarga->update($request->all());
        return response()->json(['data' => $tipoCarga], JsonResponse::HTTP_OK);
    }

    public function destroy(TipoCarga $tipoCarga): JsonResponse
    {
        $tipoCarga->update(['activo' => false]);
        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }
}

